{*************************************************************
**************************************************************
*    Ejemplo de uso de varios mtodos de TStrings            *
*   Realizado el 09 de Enero del 2002 para la artculo sobre *
*   Objetos Auxiliares Parte VI.                             *
*   Autor: Salvador Jover   mailto: dejover@eresmas.com      *
*                                                            *
*                                                            *
*   Revista Sintesis N 8   http://www.GrupoAlbor.com/       *
**************************************************************
**************************************************************}


unit CompConversor;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

const
  MAX_MASCARAS = 15;


type

  EErrorMascaras = Class(Exception);
  EInvalidIndex = Class(Exception);
  EInvalidSeleccion = Class(Exception);
  
  TValor = Record
     Concepto: String[255];
     Entrada: Double;
  end;

  PEuro = ^TEuro;   //puntero al registro

  TMoneda = Integer;

    //enumeracin (mscaras)
  TMascaras = (UN_CENT,
               DOS_CENT,
               CINCO_CENT,
               DIEZ_CENT,
               VEINTE_CENT,
               CINCUENTA_CENT,
               UN_EUROS,
               DOS_EUROS,
               CINCO_EUROS,
               DIEZ_EUROS,
               VEINTE_EUROS,
               CINCUENTA_EUROS,
               CIEN_EUROS,
               DOSCIENTOS_EUROS,
               QUINIENTOS_EUROS);


  //estructura asociada a cada una de las cadenas de la lista de Strings
  //asociamos una al componente Conversor para mantener el total acumulado
  //de desgloses y resto por desglosar
  TEuro = record
    VEuros: Double;
    VResto: Double;  //Lo que reste del cambio
    VDesglose: ARRAY [Low(TMascaras)..High(TMascaras)] of TMoneda;
  end;


  //Conjundo de mscaras: mantiene el estado de seleccin de desglose
  TSeleccion = Set of TMascaras;

  //Evento: Damos la oportunidad al usuario de que pueda hacer algn tipo
  //        de accin cada vez que se selecciona o se deselecciona una mscara
  TOnAfterChangeMascaraEvent = procedure (Sender: TObject) of Object;
  TOnAfterChangeConversorEvent = procedure (Sender: TObject) of Object;

  TConversorCambio = class(TComponent)
  private
    { Private declarations }
    fEuro: TEuro; //una variable a la estructura de totales
    fSeleccion: TSeleccion; //Seleccin actual de monedas
    fLista: TStrings; //nuestra lista de cadenas

    //facilitamos un evento que nos avise de cambios en la seleccion de mscaras
    FOnAfterChangeMascaraEvent: TOnAfterChangeMascaraEvent;
    FOnAfterChangeConversorEvent: TOnAfterChangeConversorEvent; //contador de lineas

    FCount: Integer;

    function GetName(Index: Integer): String;
    function GetValue(Index: Integer): String;
    procedure SetOnAfterChangeMascaraEvent(const Value: TOnAfterChangeMascaraEvent);
    procedure SetOnAfterChangeConversorEvent(const Value: TOnAfterChangeConversorEvent);
    function GetDesgloseCambio(Index: Integer; fMascara: TMascaras): Integer;
    function GetDesgloseTotal(fMascara: TMascaras): Integer;
  protected
    { Protected declarations }
    procedure Calcular; virtual;
    property Lista: TStrings read fLista;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); Override;
    destructor Destroy; Override;

    procedure Agregar_Mascara(Value:TMascaras); //nos permite aadir una valor al conjunto
    procedure Eliminar_Mascara(Value:TMascaras); //nos permite eliminar una valor al conjunto
    procedure Inicializar_Mascaras;
    function Buscar_Mascara(Value:TMascaras): Boolean;  //esta activa dicha mscara
    function Nominal_Mascara(Value:TMascaras): String; virtual;  //rtulo
    function Contar_Mascaras: Integer; //devuelve el nmero de mscaras activas
    function Total_Mascaras: Integer; //devuelve el nmero total de mscaras;

    procedure BorrarLista;//inicializa la lista
    function AgregarLista(const Values: Array of TValor): Boolean;

    procedure SaveToFile(const FileName: String);
    procedure LoadFromFile(const FileName: String);

    function GetTotal: String; //devuelve total suma de euros
    function GetResto: String; //devuelve total suma Resto por desglosar

    function ShowEditor(Position: Integer): Integer; virtual; //editor

    property Values[Index:Integer]: String read GetValue;
    property Names[Index: Integer]: String read GetName;
    property DesgloseCambio[Index: Integer; fMascara: TMascaras]: Integer read GetDesgloseCambio;
    property DesgloseTotal[fMascara: TMascaras]: Integer read GetDesgloseTotal;

    property Count: Integer read FCount; //nmero de elementos

    property OnAfterChangeMascaraEvent: TOnAfterChangeMascaraEvent read FOnAfterChangeMascaraEvent write SetOnAfterChangeMascaraEvent;
    property OnAfterChangeConversorEvent: TOnAfterChangeConversorEvent read FOnAfterChangeConversorEvent write SetOnAfterChangeConversorEvent;
  end;


implementation

uses Datos;

  const
  MENOR_MASCARA = Low(TMascaras);
  MAYOR_MASCARA = High(TMascaras);
  VUnidad: ARRAY[Low(TMascaras)..High(TMascaras)] of Integer = (1, 2, 5, 10, 20, 50, 1, 2, 5, 10, 20, 50, 100, 200, 500);


//--------------------------------------------------------------
// CONSTRUCTOR: Create
//
// ACCION:  A la creacin de este componente, debemos inicializar todas
//          aquellas estructuras y campos que participan.
//
constructor TConversorCambio.Create(AOwner: TComponent);
var
fMascara: TMascaras;
begin
   inherited Create(AOwner);
   fcount:= 0;
   fSeleccion:= []; // todava no hay seleccin de mscara [conjunto = vacio]
   //La clase TStrings nos permitir mantener la lista de conceptos y una
   //estructura TEuro asociada a cada una de las cadenas.
   fLista:= TStringList.Create;

   with fEuro do // inicializamos la estructura de totales
      begin
      VEuros:= 0.0;
      VResto:= 0.0;
      fMascara:= MENOR_MASCARA;
      while fMascara <= MAYOR_MASCARA do
        begin
        VDesglose[fMascara]:= 0;
        fMascara:= Succ(fMascara);
        end;
      end;
end;


//--------------------------------------------------------------
// DESTRUCTOR: Destroy
//
// ACCION:  Liberamos toda la memoria reservada y destruimos aquellos objetos
//          creados dinmicamente.
//
destructor TConversorCambio.Destroy;
begin
   BorrarLista; //liberamos todas las estructuras
   fLista.Free;//destruimos la lista
   inherited Destroy;
end;


//--------------------------------------------------------------
// METODO PUBLICO: Agregar_Mascara
//
// PARAMETROS: TMascaras (tipo enumerado que representa cada uno de los posibles
//                        desgloses. En este caso toma como valores las distintas
//                        monedas sobre las que hay que operar)
//
// ACCION:  Al agregar una mscara incluimos dicho valor en el conjunto fSeleccion,
//          que es un conjunto de TMascaras [TSeleccion = Set of TMascaras;]
//          La moneda queda seleccionada
//
procedure TConversorCambio.Agregar_Mascara(Value:TMascaras); //nos permite aadir una valor al conjunto
begin
   if not (Value in fSeleccion) then
       begin
       fSeleccion:= fSeleccion + [Value];
       if Assigned(OnAfterChangeMascaraEvent) then OnAfterChangeMascaraEvent(Self);
       end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: Eliminar_Mascara
//
// PARAMETROS: TMascaras (tipo enumerado que representa cada uno de los posibles
//                        desgloses. En este caso toma como valores las distintas
//                        monedas sobre las que hay que operar)
//
// ACCION:  Eliminamos una mscara del conjunto de fSeleccion.
//          La moneda es excluida de la seleccin
//
procedure TConversorCambio.Eliminar_Mascara(Value:TMascaras); //nos permite aadir una valor al conjunto
begin
   if (Value in fSeleccion) then
      begin
      fSeleccion:= fSeleccion - [Value];
      if Assigned(OnAfterChangeMascaraEvent) then OnAfterChangeMascaraEvent(Self);
      end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: Buscar_Mascara
//
// PARAMETROS: TMascaras (tipo enumerado que representa cada uno de los posibles
//                        desgloses. En este caso toma como valores las distintas
//                        monedas sobre las que hay que operar)
//
// ACCION:  Devuelve True si la mscara esta en el conjunto y false en caso
//          contrario.
//
function TConversorCambio.Buscar_Mascara(Value:TMascaras): Boolean;
begin
   Result:= (Value in fSeleccion);//esta activa dicha mscara
end;


//--------------------------------------------------------------
// METODO PUBLICO: Contar_Mascaras
//
// PARAMETROS: No
//
// VALOR RETORNO: Integer
//
// ACCION:  Cuenta el total de mscaras activas en el conjunto fSeleccion
//
function TConversorCambio.Contar_Mascaras: Integer;
   var
   xMascara: TMascaras;
begin
   xMascara:= UN_CENT; //nos situamos en la primera de las mscaras
   Result:= 0;
   while xMascara <= QUINIENTOS_EUROS do //y las recorremos todas
      begin //si est activa
      if Buscar_Mascara(xMascara) then Result:= Result + 1; //la contamos
      xMascara:= Succ(xMascara); //siguiente mscara
      end;
end;

//--------------------------------------------------------------
// METODO PUBLICO: Nominal_Mascara
//
// PARAMETROS: TMascaras (tipo enumerado que representa cada uno de los posibles
//                        desgloses. En este caso toma como valores las distintas
//                        monedas sobre las que hay que operar)
//
// VALOR RETORNO: String
//
// ACCION:  Nos devuelve un rtulo asociado a cada mscara.
//
function TConversorCambio.Nominal_Mascara(Value:TMascaras): String;
begin
   Result:= '';
   case Value of
      UN_CENT:             Result:='1 Ctms.';
      DOS_CENT:            Result:='2 Ctms.';
      CINCO_CENT:          Result:='5 Ctms.';
      DIEZ_CENT:           Result:= '10 Ctms.';
      VEINTE_CENT:         Result:= '20 Ctms.';
      CINCUENTA_CENT:      Result:= '50 Ctms.';
      UN_EUROS:            Result:= '1 Euro.';
      DOS_EUROS:           Result:= '2 Euros.';
      CINCO_EUROS:         Result:= '5 Euros.';
      DIEZ_EUROS:          Result:= '10 Euros.';
      VEINTE_EUROS:        Result:= '20 Euros.';
      CINCUENTA_EUROS:     Result:= '50 Euros.';
      CIEN_EUROS:          Result:= '100 Euros.';
      DOSCIENTOS_EUROS:    Result:= '200 Euros.';
      QUINIENTOS_EUROS:    Result:= '500 Euros.';
   end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: BorrarLista
//
// PARAMETROS: No
//
// ACCION:  Inicializamos la lista de cadenas
//          Liberamos todas la memoria de las estructuras TEuro asociadas a
//          la lista
//
procedure TConversorCambio.BorrarLista;
   var
   xIndice: Integer;
   Nodo: PEuro;
begin
   for xIndice:= 0 to Lista.Count-1 do
      begin
      Nodo:= PEuro(Lista.Objects[xIndice]);
      Dispose(Nodo);
      Lista.Objects[xIndice]:= Nil;
      end;
   Lista.Clear;
   fCount:= 0;
end;


//--------------------------------------------------------------
// METODO PUBLICO: AgregarLista
//
// PARAMETROS: Array of TValor (Entrada)
               {Un array dinmico nos permite recoger los parmetros necesarios
               para el calculo: Concepto y Valor}
//
//             Array of Integer (Salida)
               {Un array dinmico nos permite exportar el resultado de los
               clculos}
               {EJEMPLO:
               MASCARAS ACTIVAS     [UN_EUROS, DOS_EUROS, CINCO_EUROS]
               ENTRADA..............CONCEPTO("Factura N22 de Juan Antonio")
                                    VALOR(228,2)
               SALIDA...............(45,1,1)}
//
// VALOR RETORNO: TEuro
               {En este registro acumularemos los totales de los desgloses
               El campo VResto recoge lo que ya no puede ser desglosado segn
               la seleccin activa
               En el ejemplo anterior
               VResto = 0,2}
//
// ACCION:  Clculo de los desgloses segn la seleccin
            {Cada vez que se ejecuta este procedimiento se inicializa la lista y
            se recalculan los valores de salida}
//
//
function TConversorCambio.AgregarLista(const Values: Array of TValor):Boolean;
   var
   xIndice: Integer;
   s: String;
   f: Double;
   DataEuro: PEuro;
   fMascara: TMascaras;
begin
   Result:= False;
   BorrarLista; //inicializamos la lista
   try
   //recorremos el array de entrada de datos
   for xIndice:=Low(Values) to High(Values) do
      begin
      s:= TValor(Values[xIndice]).Concepto; //almacenamos temporalmente el concepto!
      f:= TValor(Values[xIndice]).Entrada;  //almacenamos temporalmente el valor-! !
                                                                                {! !}
      New(DataEuro);                                                            {! !}
      with DataEuro^do                                                          {! !}
         begin                   {y es guardado finalmente en el registro}      {! !}
         VEuros:= f;   //--------------------------------------------------------! !}
         VResto:= 0.0;
         fMascara:= MENOR_MASCARA;
         while fMascara <= MAYOR_MASCARA do
            begin
            VDesglose[fMascara]:= 0;
            fMascara:= Succ(fMascara);
            end;                                                            {!}
         end;
                            {es aadida cadena y objeto a la lista}           {!}
      Lista.AddObject(s, TObject(Dataeuro));//-------------------------------------!
      Inc(fCount);
      end;
   Calcular; //ya podemos efectuar las operaciones necesarias
   Result:= True; //devolvemos la estructura de totales
   Except
   on E:Exception do
     if Assigned(OnAfterChangeConversorEvent) then OnAfterChangeConversorEvent(Self);
   //Actualizamos el interfaz
   end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: Calcular
//
// PARAMETROS: Array of Integer (Salida)
               {Un array dinmico nos permite exportar el resultado de los
               clculos}
               {EJEMPLO:
               MASCARAS ACTIVAS     [UN_EUROS, DOS_EUROS, CINCO_EUROS]
               ENTRADA..............CONCEPTO("Factura N22 de Juan Antonio")
                                    VALOR(228,2)
               SALIDA...............(45,1,1)}
//
// ACCION:  Operatoria principal
//          El objetivo final es estregar la lista de desgloses dentro del
//          array dinmico
//
procedure TConversorCambio.Calcular;
var
xIndice, zIndice: Integer;
Valor_e, Valor_c: Integer;
fMascara: TMascaras;
begin
   //nos aseguramos que la lista contiene algn valor
   if Lista.Count <= 0 then
     begin  //tenemos que dejar que el usuario actualice el interfaz
     if Assigned(OnAfterChangeConversorEvent) then OnAfterChangeConversorEvent(Self);
     exit;
     end;
   //inicializamos la estructura de totales
   with fEuro do
      begin
      VEuros:= 0.0;
      VResto:= 0.0;
      fMascara:= MENOR_MASCARA;
      while fMascara <= MAYOR_MASCARA do
         begin
         fEuro.VDesglose[fMascara]:= 0;
         fMascara:= Succ(fMascara);
         end;
      end;

   //para cada una de las cadenas de la lista (y sus objetos asociados)
   for xIndice:= 0 to Lista.Count-1 do
      begin
      //obtenemos el valor sobre el que haremos el desglose
      Valor_e:= Trunc(PEuro(Lista.Objects[xIndice])^.VEuros);
      Valor_c:= StrToInt(FloatToStr(Round((PEuro(Lista.Objects[xIndice])^.VEuros - Valor_e)*100)));
      fEuro.VEuros:= fEuro.VEuros + PEuro(Lista.Objects[xIndice])^.VEuros; //y acumulamos el total de euros


      {El proceso de desglose es muy sencillo:
      Tomado el total a desglosar, nos situamos en la moneda activa mayor y
      sustraemos progresivamente su valor del total.
      EJEMPLO:
      MASCARAS ACTIVAS............[DOSCIENTOS_EUROS, VEINTE_EUROS, CINCO EUROS]
      Valor Total.................422,2 Euros
                                                 Array = (0,0,0)
      Activa QUINIENTOS_EUROS?   NO  ---------> Siguiente
      Activa DOSCIENTOS_EUROS?   SI  ---------> Array = (2,0,0)
      Activa CIEN_EUROS?         NO  ---------> Siguiente
      Activa CINCUENTA_EUROS?    NO  ---------> Siguiente
      Activa VEINTE_EUROS?       SI  ---------> Array = (2,1,0)
      Activa DIEZ_EUROS?         NO  ---------> Siguiente
      Activa CINCO_EUROS?        SI  ---------> Array = (2,1,0)
      Activa DOS_EUROS?          NO  ---------> Siguiente
      Activa UN_EUROS?           NO  ---------> Siguiente
      Activa CINCUENTA_CENTIMOS? NO  ---------> Siguiente
      Activa VEINTE_CENTIMOS?    NO  ---------> Siguiente
      Activa DIEZ_CENTIMOS?      NO  ---------> Siguiente
      Activa CINCO_CENTIMOS?     NO  ---------> Siguiente
      Activa DOS_CENTIMOS?       NO  ---------> Siguiente
      Activa UN_CENTIMOS?        NO  ---------> Siguiente
      Al finalizar el proceso estan acumulados los totales en la estructura de
      totales y el campo VResto nos entregara el valor 2,2 Euros}
     zIndice:= Integer(TMascaras(MAYOR_MASCARA));
     While zIndice >= 0 DO
          begin
          fMascara:= TMascaras(zIndice);
          //empezamos a filtrar las condiciones
          if Buscar_Mascara(fMascara) then
               begin
               if fMascara >= UN_EUROS then
                    begin
                    While Valor_e >= VUnidad[fMascara] do
                         begin
                         Inc(PEuro(Lista.Objects[xIndice])^.VDesglose[fMascara]); //desglose acumulado
                         Valor_e:= Valor_e - VUnidad[fMascara]; //seguimos operando?: si cumple condicin
                         end;
                    end
               else
                    begin
                    if Valor_e > 0 then
                         begin
                         Valor_c:= Valor_c + Valor_e * 100;
                         Valor_e:= 0;
                         end;
                    While Valor_c >= VUnidad[fMascara] do
                         begin
                         Inc(PEuro(Lista.Objects[xIndice])^.VDesglose[fMascara]);
                         Valor_c:= Valor_c - VUnidad[fMascara];
                         end;
                    end;
               //acumulamos el total
               fEuro.VDesglose[fMascara]:= fEuro.VDesglose[fMascara] + PEuro(Lista.Objects[xIndice])^.VDesglose[fMascara];
               end;
          Dec(zIndice);
          end;//endbegin while

       if Valor_e > 0 then fEuro.VResto:= fEuro.VResto + StrToFloat(IntToStr(Valor_e));
       if Valor_c > 0 then fEuro.VResto:= fEuro.VResto + (Valor_c / 100);
       //hemos acumulado el valor de lo que no ha sido desglosado en la estructura
       //de totales
       end;
   if Assigned(OnAfterChangeConversorEvent) then OnAfterChangeConversorEvent(Self);
end;


procedure TConversorCambio.SetOnAfterChangeMascaraEvent(
  const Value: TOnAfterChangeMascaraEvent);
begin
  FOnAfterChangeMascaraEvent := Value;
end;

procedure TConversorCambio.SetOnAfterChangeConversorEvent(
  const Value: TOnAfterChangeConversorEvent);
begin
  FOnAfterChangeConversorEvent := Value;
end;


//--------------------------------------------------------------
// METODO PUBLICO: SaveToFile
//
// PARAMETROS: Filename: String  <El nombre del fichero>
//
// ACCION:  Implementamos un mtodo para guardar en un fichero el contenido
//          actual del componente Conversor
//
            {Vamos a aprovechar la capacidad de leer lineas del tipo
             CONCEPTO = VALOR en la clase TStrings para guardar el
             contenido del fichero segn ese patrn, incluida una
             primera linea donde se almacenan las mascaras elegidas }
//
procedure TConversorCambio.SaveToFile(const FileName: String);
var
StringTemp: String;
ListaTemp: TStrings;
S: Array [0..MAX_MASCARAS-1] of Char;
xIndice: Integer;
begin
   ListaTemp:= TStringList.Create; //nos valemos de una lista temporal
   try
      //vamos a exigirle al conversor que al menos exista una seleccion de 2 mscaras
      if Contar_Mascaras < 2 then
          Raise EInvalidSeleccion.Create('Seleccion de Mscaras debe ser mayor de dos');
      //tomaremos todo el contenido del miembro lista del conversor
      //incluidos los objetos. Estos nos serviran para recuperar los
      //valores asociados a cada lista
      ListaTemp.Assign(Lista);
      for xIndice:= 0 to MAX_MASCARAS-1 do //recorremos todas las mscaras
        if Buscar_Mascara(TMascaras(xIndice)) then S[xIndice]:= '1'
        else S[xIndice]:= '0';
      StringTemp:= 'MASCARA='+String(s); //ya tenemos la primera lnea
      ListaTemp.Insert(0,StringTemp); //la instamos en primer lugar
      //Y recorremos el resto de lista para construir el resto de igualdades
      for xIndice:= 1 to ListaTemp.Count - 1 do
         if ListaTemp.Strings[xIndice] <> '' then
            ListaTemp.Strings[xIndice]:= ListaTemp.Strings[xIndice] +  '=' +
               FormatFloat('#0.00', PEuro(ListaTemp.Objects[xIndice])^.VEuros);
      ListaTemp.SaveToFile(Filename);//guardamos el fichero
   finally
      ListaTemp.Free;//y liberamos la lista temporal
   end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: LoadFromFile
//
// PARAMETROS: Filename: String  <El nombre del fichero>
//
// ACCION:  Implementamos un mtodo para cargar el contenido de nuestro
//          componente Conversor con el fichero de igualdades.
//
            {Al guardar hicimos corresponder el contenido del componente
             conversor con igualdades del tipo CONCEPTO = VALOR, que son
             legibles en la clase TStrings. Nos puede servir para ver
             como pudieran ser utilizadas propiedades de tipo matricial
             como Values o Names, que nos ayudan en este tipo de igualdades.}
//
procedure TConversorCambio.LoadFromFile(const FileName: String);
var
ListaTemp: TStrings;
S: String [MAX_MASCARAS];
xIndice: Integer;
DataEuro: PEuro;
fMascara: TMascaras;
begin
   ListaTemp:= TStringList.Create; //necesitamos una lista temporal
                                   //para leer las igualdades
   try
      ListaTemp.LoadFromFile(Filename);//procedemos a cargar el fichero en la lista temporal
      //leemos el contenido del valor de la primera linea
      //que no es otro que la seleccin de mscaras almacenada en el fichero
      // Ejemplo: MASCARA=10010001110101
      // Un valor 1 nos indica que est activada y 0 que esta desactivada
      S:=ListaTemp.Values[ListaTemp.Names[0]];
      fSeleccion:= []; //inicializamos la seleccion de mscaras
      //recorremos el string comprobando si esta activada o no
      for xIndice:= 0 to MAX_MASCARAS - 1 do
         if S[xIndice+1] = '1' then
            fSeleccion:= fSeleccion + [TMascaras(xIndice)] //activamos la mscara
         else  //si no est activada la desactivamos
            if S[xIndice+1] = '0' then fSeleccion:= fSeleccion - [TMascaras(xIndice)]
            else Raise EErrorMascaras.Create('Error: formato no valido de archivo');
      BorrarLista;//ya podemos borrar el contenido de la lista de conversor
      //disponemos a inicializar la lista temporal con los valores recogidos
      //del fichero, creando la estructura de objetos
      for xIndice:= 1 to ListaTemp.Count - 1 do
         begin
         New(DataEuro);  //reservamos memoria
         with DataEuro^do
               begin                   {y es guardado finalmente en el registro}
               VEuros:= StrToFloat(ListaTemp.Values[ListaTemp.Names[xIndice]]);
               VResto:= 0.0;
               fMascara:= MENOR_MASCARA;
               while fMascara <= MAYOR_MASCARA do
                    begin
                    VDesglose[fMascara]:= 0;
                    fMascara:= Succ(fMascara);
                    end;                                                            {!}
               end;                   { y es aadida cadena y objeto a la lista}        {!}
         Lista.AddObject(ListaTemp.Names[xIndice], TObject(Dataeuro));
         Inc(fCount); {incrementamos el contador de lineas}
         end;
   finally
      ListaTemp.Free;
      Calcular; //nos permitira crear la estructura de objetos y de totales
    end;
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetName
//
// PARAMETROS: Index: Integer
//
// ACCION:  Devolvemos la cadena segun el indice que ocupa en la matriz
//          Implementamos un mtodo para poder acceder a los valores de
//          la lista: El metodo agregar nos permite comunicar de dentro
//          a fuera. Este mtodo nos permite comunicar de forma inversa,
//          de fuera a dentro.
function TConversorCambio.GetName(Index: Integer): String;
begin
     Result:= '';
     if (Index < 0) and (Index > Lista.Count - 1) then
          Raise EInvalidIndex.Create('Error: Indice no vlido');
     Result:= Lista.Strings[Index];
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetValue
//
// PARAMETROS: Index: Integer
//
// ACCION:  Devolvemos la valor VEuros, que es miembro del objeto
//          asociado a una cadena determinada.
//          Este valor nos interesa para poder comunicarnos con
//          el objeto conversor ya que no podemos acceder a la lista
//          pues queda protegida de la manipulacin.
function TConversorCambio.GetValue(Index: Integer): String;
begin
     Result:= '';
     if (Index < 0) and (Index > Lista.Count - 1) then
          Raise EInvalidIndex.Create('Error: Indice no vlido');
     Result:= FormatFloat('#0.00',PEuro(Lista.Objects[Index])^.VEuros);
end;


//--------------------------------------------------------------
// METODO PUBLICO: Inicializar_Mascaras
//
// PARAMETROS: No tiene
//
// ACCION: Desactivamos todas las mscaras
//         Este procedimiento es pblico.
//
procedure TConversorCambio.Inicializar_Mascaras;
begin
     fSeleccion:= [];
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetDesgloseCambio
//
// PARAMETROS: index: integer-------> elemento sobre que queremos leer
//             fMascara: TMascaras--> desglose elegido en el elemento a leer
//
// ACCION: Obtenemos los valores actuales del conversor respecto a
//         los desgloses obtenidos. Index nos permite poder acceder
//         a cada uno de los elementos de la lista interna que
//         mantiene el conversor.
//         Se utiliza para actualizar el interfaz grfico
//
function TConversorCambio.GetDesgloseCambio(Index: Integer;
  fMascara: TMascaras): Integer;
begin
     if (Index < 0) and (Index > Lista.Count - 1) then
          Raise EInvalidIndex.Create('Error: Indice no vlido');
     Result:= PEuro(Lista.Objects[Index])^.VDesglose[fMascara];
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetDesgloseTotal
//
// PARAMETROS: fMascara: TMascaras
//
// ACCION: Obtenemos el valor de desglose segn la unidad monetaria
//         elegida para leer
//
function TConversorCambio.GetDesgloseTotal(fMascara: TMascaras): Integer;
begin
     Result:= fEuro.VDesglose[fMascara];
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetTotal
//
// PARAMETROS: No tiene
//
// ACCION: Obtenemos la suma total de Euros que almacena el Conversor
//
function TConversorCambio.GetTotal: String;
begin
     Result:= FormatFloat('#0.00', fEuro.VEuros);
end;


//--------------------------------------------------------------
// METODO PUBLICO: GetResto
//
// PARAMETROS: No tiene
//
// ACCION: Obtenemos la suma total Resto que almacena el Conversor
//
function TConversorCambio.GetResto: String;
begin
     Result:= FormatFloat('#0.00', fEuro.VResto);
end;


//--------------------------------------------------------------
// METODO PUBLICO: Total_Mascaras
//
// PARAMETROS: No tiene
//
// ACCION: El valor superior en el tipo TMascaras
//
function TConversorCambio.Total_Mascaras: Integer;
begin
     Result:= Integer(High(TMascaras));
end;


//--------------------------------------------------------------
// METODO PUBLICO: ShowEditor
//
// PARAMETROS: Position: Integer--------> Itemindex del Editor
//
// ACCION: Invocamos al editor de lineas. El modulo Datos.pas
//         implementa el editor que hemos elegido.
//         Este mdulo se relaciona UNICAMENTE con el componente
//         conversor y no tiene nada relacin alguna con el
//         interfaz grfico.
//         En un descendiente del Conversor podramos facilmente
//         redefinir este mtodo e invocar otro editor construido
//         con otros fines.
//
function TConversorCambio.ShowEditor(Position: Integer): Integer;
var
Dat: TfrmDatos;
begin
     Dat:= TfrmDatos.Create(Self, (count > 0), Position);
     try
          if Dat.ShowModal = mrOk then BorrarLista;
          Result:= Dat.stg_Tabla.Row;
     finally
     Dat.Free;
     end;
end;

end.
