unit UColaImpresion;

interface

uses
  Windows, SysUtils, Classes, UShellNotify;


const
  //extension del fichero que contiene los parametros
  EXTENSION = 'SJC';


type
  TColaImpresion = class;

  THiloProceso = class(TThread)
  private
    FColaImpresion: TColaImpresion;
    procedure DoProcesar;
  protected
    procedure Execute; override;
  public
    constructor Create(CreateSuspended: Boolean;
     AColaImpresion: TColaImpresion);
    destructor Destroy; override;
  end;

  TTrabajo = class
  private
    FIDSesion: Integer;
    FCadena: String;
    FTaskName: String;
    FErrorMessage: String;
    FFilename: TFileName;
    procedure SetIDSesion(const Value: Integer);
    procedure SetTaskName(const Value: String);
    procedure SetErrorMessage(const Value: String);
    procedure SetFilename(const Value: TFileName);
  protected
  public
    constructor Create(const ACadena: String);
    function LeeCampo: String;
    property Cadena: String read FCadena;
    property IDSesion: Integer read FIDSesion write SetIDSesion;
    property TaskName: String read FTaskName write SetTaskName;
    property ErrorMessage: String read FErrorMessage write SetErrorMessage;
    property Filename: TFileName read FFilename write SetFilename;
  end;

  TNotifyTaskEvent = procedure(Sender: TObject; ATrabajo: TTrabajo) of object;
  TNotifyDoHacerEvent = procedure(Sender: TObject; ATrabajo: TTrabajo;
   AModulo, AFuncion: Integer) of object;

  TColaImpresion = class(TComponent)
  private
    FColaImpresion: TStrings;
    FOnAddTaskEvent: TNotifyTaskEvent;
    FOnDeleteTaskEvent: TNotifyTaskEvent;
    FOnDoHacerEvent: TNotifyDoHacerEvent;
    FPath: String;
    NotifyHandle: HWND;
    Notificador: TShellNotification;
    Hilo: THiloProceso;
    FActive: Boolean;
    function Get(Index: Integer): string;
    procedure SetOnAddTaskEvent(const Value: TNotifyTaskEvent);
    procedure SetOnDeleteTaskEvent(const Value: TNotifyTaskEvent);
    procedure SetOnDoHacerEvent(const Value: TNotifyDoHacerEvent);
    procedure SetPath(const Value: String);
    procedure SetActive(const Value: Boolean);
    procedure OnCreateFileEvent(Sender: TObject; Path: String);
    { Private declarations }
  protected
    procedure DoAddTaskEvent(ATrabajo: TTrabajo); virtual;
    procedure DoDeleteTaskEvent(ATrabajo: TTrabajo); virtual;
    procedure DoHacerTaskEvent(Sender: TObject; ATrabajo: TTrabajo;
    AModulo, AFuncion: Integer); virtual;
    function DoLinkFileCreateEvent(Sender: TObject;
     Path: String): TShNotificationEvent; virtual;
    procedure Delete(Index: Integer);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    Destructor Destroy; override;
    procedure Add(const ATextoAMostrar: String; ATrabajo: TTrabajo);
    function Count: Integer;
    function ExtraeItem: TTrabajo;

    procedure VaciarCola;  //ejecuta todos los trabajos
    procedure UpdateCola;  //adjunta a la cola los ficheros leidos en busqueda
    procedure DoHacer;     //ejecuta el trabajo actual
    procedure LeeFicheroImpresion(const AFileName: String); //lee fichero

    property Strings[Index: Integer]: string read Get; default;
    property Path: String read FPath write SetPath;         //ruta lectura
    property Active: Boolean read FActive write SetActive;
    property OnAddTaskEvent: TNotifyTaskEvent read FOnAddTaskEvent write SetOnAddTaskEvent;
    property OnDeleteTaskEvent: TNotifyTaskEvent read FOnDeleteTaskEvent write SetOnDeleteTaskEvent;
    property OnDoHacerEvent: TNotifyDoHacerEvent read FOnDoHacerEvent write SetOnDoHacerEvent;
  end;


implementation

uses UFuncionesAuxiliares;



// Constructor
//
// Objetivo: Creamos la lista que guardar las tareas
//
//
constructor TColaImpresion.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FActive:= False;
  FColaImpresion:= TStringList.Create;
  Hilo:= THiloProceso.Create(True, self);
  Hilo.Start;
end;


// Procedimiento Add
//
// Objetivo: Aade una tarea a la cola de tareas, objeto y texto generico tarea
//           Adicionalmente generamos el evento para dar respuesta al suceso
//
procedure TColaImpresion.Add(const ATextoAMostrar: String; ATrabajo: TTrabajo);
begin
  with FColaImpresion do
     DoAddTaskEvent(TTrabajo(Objects[AddObject(ATextoAMostrar, ATrabajo)]));
end;

// Funcion
//
// Objetivo: Devuelve la cantidad de tareas pendientes
//
//
function TColaImpresion.Count: Integer;
begin
  Result:= FColaImpresion.Count;
end;


// Procedimento/Funcion
//
// Objetivo:
//
//
procedure TColaImpresion.Delete(Index: Integer);
begin
  DoDeleteTaskEvent(TTrabajo(FColaImpresion.Objects[Index]));
  FColaImpresion.Delete(Index);
end;


// Destructor
//
// Objetivo: Eliminamos la cola de tareas y los objetos existentes en la misma
//
//
destructor TColaImpresion.Destroy;
var
  i: Integer;
begin
  if FActive then Active:= False;

  Hilo.Terminate;

  for i:= 0 to FColaImpresion.Count-1 do TTrabajo(FColaImpresion.Objects[i]).Free;
  FColaImpresion.Clear;
  FColaImpresion.Free;
  FColaImpresion:= nil;

  inherited Destroy;
end;


// Procedimento/Funcion
//
// Objetivo: Disparo del evento al aadir una tarea, si est asignado.
//
//
procedure TColaImpresion.DoAddTaskEvent(ATrabajo: TTrabajo);
begin
  if Assigned(FOnAddTaskEvent) then FOnAddTaskEvent(Self, ATrabajo);
end;


// Procedimento
//
// Objetivo: Disparo del evento al eliminar una tarea, si est asignado
//
  {
    La eliminacn de una tarea puede ocurrir tanto si es procesada como
    si el usuario decidiera cancelarla, en el caso de existir en la cola de
    tareas y no estar en proceso
  }
//
procedure TColaImpresion.DoDeleteTaskEvent(ATrabajo: TTrabajo);
begin
  if Assigned(FOnDeleteTaskEvent) then FOnDeleteTaskEvent(Self, ATrabajo);
end;


function TColaImpresion.DoLinkFileCreateEvent(Sender: TObject; Path: String): TShNotificationEvent;
begin
  Result:= OnCreateFileEvent;
end;

// Procedimiento Hacer
//
// Objetivo: Extrae de la cola de procesos el mas antiguo.
//
    {
      La tarea tiene un procedimiento LeeCampo, que nos permite desmenuzar
      la cadena de parametros. Cada vez que es invocado, nos comemos parte de la
      cadena, obteniendo un nuevo parametro.
    }
//
procedure TColaImpresion.DoHacer;
var
  FTarea: TTrabajo;
  Funcion: Integer;
  Modulo: Integer;
begin
  if not FActive then Exit;

  FTarea:= ExtraeItem;
  try
    //si no hay nada que imprimir nos salimos
    if FTarea = Nil then Exit;

    //obtenermos el modulo que tiene que imprimir/procesar y otros parametros necesarios
    //por cada lectura extrae de la cadena lo ledo
    if not TryStrToInt(FTarea.LeeCampo, Modulo) then
      raise Exception.Create('Campo mdulo es requerido y no se define');

    if not TryStrToInt(FTarea.LeeCampo, Funcion) then
      raise Exception.Create('Campo funcin es requerido y no se define');

    DoHacerTaskEvent(self, FTarea, Modulo, Funcion);
  finally
    FTarea.Free;
  end;
end;


procedure TColaImpresion.DoHacerTaskEvent(Sender: TObject; ATrabajo: TTrabajo; AModulo,
  AFuncion: Integer);
begin
  if Assigned(FOnDoHacerEvent)  then FOnDoHacerEvent(Sender, ATrabajo, AModulo, AFuncion);
end;

// Funcion ExtraeItem
//
// Objetivo: Extrae de la cola la tarea mas antigua si existe alguna
//
//
function TColaImpresion.ExtraeItem: TTrabajo;
begin
  if Count = 0 then
    Result:= Nil
  else begin
    Result:= TTrabajo(FColaImpresion.Objects[0]);
    Delete(0);
  end;
end;

// Funcion Get
//
// Objetivo: Retorna la cadena generica que identifica la tarea
//
//
function TColaImpresion.Get(Index: Integer): string;
begin
  Result:= FColaImpresion[Index];
end;


// Procedimiento LeeFicheroImpresion
//
// Objetivo: Lee un fichero y lo convierte en un proceso o tarea
//
    {
      El fichero contiene toda la informacin necesaria. Hay parte de datos
      que no son propios del proceso sino del contexto de la comunicacin,
      por lo que no son incluidos en la misma cadena de parametros, ya
      que al ser leidos mediante LeeCampo se excluyen previo a ser incluidas
      en la instancia ColaDeImpresion.
      Finalmente se borra fsicamente el fichero para que no pueda ser
      leido nuevamente. De esa forma se evita caer en un bucle infinito si
      existe algun sistema que monitoriza la lectura mediante un timer a modo
      de spool.
    }
//
procedure TColaImpresion.LeeFicheroImpresion(const AFileName: String);
var
  sParam: String;
  FTask: TTrabajo;
  fIDSesion: Integer;
begin
  //leemos el fichero
  sParam:= LoadFile(AFilename);
  try
    FTask:= TTrabajo.Create(sParam);    //creamos la tarea
    try
      FTask.Filename:= AFileName;         //guardamos el nombre del fichero

      //obtenemos el idsesion si existe -parametros de seguridad-
      if TryStrToInt(FTask.LeeCampo, fIDSesion) then FTask.IDSesion:= fIDSesion;
      //guardamos el nombre de la tarea o proceso (podria ser el nombre del informe)
      FTask.TaskName:= FTask.LeeCampo;
      //aadimos finalmente la tarea a la cola de tareas
      Add(AFilename, FTask);
    except
      FTask.Free;
      Raise;
    end;
  finally
    DeleteFile(AFileName);  //finalmente borramos el fichero
  end;
end;


// Metodos escritura
//
// Objetivo: Metodo de escritura evento
//
//
procedure TColaImpresion.SetActive(const Value: Boolean);
begin
  if Value <> FActive then
  begin
    //si estamos desactivando el componente
    //destruimos la notificacion
    case Value of
      False: begin
               if Assigned(Notificador) then FreeAndNil(Notificador);
             end;
      True: begin
                if (Trim(FPath) <> '') and DirectoryExists(FPath) then
                begin
                  Notificador:= TShellNotification.Create(self, FPath);
                  with Notificador do begin
                    OnFileCreate:= DoLinkFileCreateEvent(Notificador, FPath);//func;
                    Active:= True;
                  end;
                end
                else raise Exception.Create('No fue posible activar, verifique que la ruta existe y es vlida');
            end;
    end;
    //finalmente asignamos la propiedad si no ha existido error
    FActive := Value;
    if FActive then begin
      UpdateCola;
      VaciarCola;
    end;
  end;
end;

procedure TColaImpresion.SetOnAddTaskEvent(const Value: TNotifyTaskEvent);
begin
  FOnAddTaskEvent := Value;
end;


// Metodos escritura
//
// Objetivo: Metodo de escritura evento
//
//
procedure TColaImpresion.SetOnDeleteTaskEvent(const Value: TNotifyTaskEvent);
begin
  FOnDeleteTaskEvent := Value;
end;

procedure TColaImpresion.SetOnDoHacerEvent(const Value: TNotifyDoHacerEvent);
begin
  FOnDoHacerEvent := Value;
end;

// Procedimiento SetPath
// Metodo de escritura de la propiedad Path
//
// Objetivo: Guarda la ruta que vamos a monitorizar
//
procedure TColaImpresion.SetPath(const Value: String);
begin
  if Value <> FPath then
  begin
    FPath := Value;
    if FActive then Active:= False;
  end;
end;

// Procedimiento VaciarCola
//
// Objetivo: Mientras existan tareas las procesa (DoHacer)
//
procedure TColaImpresion.VaciarCola;
begin
  while Count > 0 do DoHacer;
end;

// Procedimiento UpdateCola
//
// Objetivo: Buscar en una ruta prefijada los archivos para su proceso a travs
//           del procedimiento LeeFicheroImpresion
//
procedure TColaImpresion.UpdateCola;
var
  SearchRec: TSearchRec;
begin
   if FindFirst(Path+'\*.'+EXTENSION, faAnyFile and not
      (faVolumeID or faDirectory), SearchRec) = 0 then
   begin
     repeat
       // Aqu procesamos cada archivo encontrado
       // Sus datos se encuentran en SearchRec.
       LeeFicheroImpresion(Path+ SearchRec.Name);
     until FindNext(SearchRec) <> 0;
     FindClose(SearchRec);
   end;
end;

// Procedimiento de tipo TShNotificationEvent
//
// Objetivo: Cada vez que sea aadido un fichero en la ruta establecida
//           damos respuesta al evento.
//           En este caso leemos el fichero y lo introducimos en la cola
//           de procesos
//
procedure TColaImpresion.OnCreateFileEvent(Sender: TObject; Path: String);
begin
  LeeFicheroImpresion(Path);
end;



{ TTrabajo }

// Constructor
//
// Objetivo: Creamos la lista que guardar las tareas
//
//
constructor TTrabajo.Create(const ACadena: String);
begin
  FCadena:= ACadena;
end;

// Funcion
//
// Objetivo: Creamos la lista que guardar las tareas
//
//
function TTrabajo.LeeCampo: String;
begin
  Result:= UFuncionesAuxiliares.LeeCampo(FCadena);
end;

// Metodos escritura
//
// Objetivo: Metodo escritura propiedad ErrorMessage
//
//
procedure TTrabajo.SetErrorMessage(const Value: String);
begin
  FErrorMessage := Value;
end;

// Metodos escritura
//
// Objetivo: Metodo escritura propiedad Filename
//
//
procedure TTrabajo.SetFilename(const Value: TFileName);
begin
  FFilename := Value;
end;

// Metodos escritura
//
// Objetivo: Metodo escritura propiedad IDSesion
//
//
procedure TTrabajo.SetIDSesion(const Value: Integer);
begin
  FIDSesion := Value;
end;

// Metodos escritura
//
// Objetivo: Metodo escritura propiedad TaskName
//
//
procedure TTrabajo.SetTaskName(const Value: String);
begin
  FTaskName := Value;
end;


{ THiloProceso }

constructor THiloProceso.Create(CreateSuspended: Boolean;
  AColaImpresion: TColaImpresion);
begin
  FColaImpresion:= AColaImpresion;
  inherited Create(CreateSuspended);
  FreeOnTerminate := True;
end;

destructor THiloProceso.Destroy;
begin
  FColaImpresion:= Nil;
  inherited;
end;

procedure THiloProceso.DoProcesar;
begin
  if Assigned(FColaImpresion) then FColaImpresion.DoHacer
  else Terminate;
end;

procedure THiloProceso.Execute;
begin
  while not Terminated do begin
     Synchronize(DoProcesar);
     Sleep(1000);
  end;
end;

end.
