unit UEstructuras;

interface

uses System.SysUtils, System.Types, System.UITypes, System.Classes,
System.Variants, FMX.TreeView, UInterfaces, FMX.Types, FMX.Dialogs;

type
  TClaseBase = class(TFMXObject)
  private
    FNodo: TTreeViewItem;
    function GetNodo: TTreeViewItem;
  protected
    function HasNodo: Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Nodo: TTreeViewItem read GetNodo;
  end;

  TComponente = class;

  AClaseBase = class of TClaseBase;

  TProducto = Class(TClaseBase, IValorable)
  private
    FID: Integer;
    FNombre: String;
    FCantidad: Double;
    FPrecio: Double;
    FValor: Double;
    FParent: TComponente;
    FExistenCambiosPendientes: Boolean;
    procedure SetParent(const Value: TComponente);
    procedure SetExistenCambiosPendientes(const Value: Boolean);
  protected
    //getters and setters
    function GetCantidad: Double; virtual;
    function GetID: Integer; virtual;
    function GetNombre: String; virtual;
    function GetPrecio: Double; virtual;
    function GetValor: Double; virtual; abstract;
    procedure SetCantidad(const Value: Double); virtual;
    procedure SetID(const Value: Integer); virtual;
    procedure SetNombre(const Value: String); virtual;
    procedure SetPrecio(const Value: Double); virtual;
    procedure ActualizaTextoNodo; virtual;
    procedure ActualizaValor; virtual; abstract;
    procedure DoDelete; virtual; abstract;
    property ExistenCambiosPendientes: Boolean read FExistenCambiosPendientes
     write SetExistenCambiosPendientes;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Delete;
    //public properties
    property ID: Integer read GetID write SetID;
    property Nombre: String read GetNombre write SetNombre;
    property Cantidad: Double read GetCantidad write SetCantidad;
    property Precio: Double read GetPrecio write SetPrecio;
    property Valor: Double read GetValor;
    property Parent: TComponente read FParent write SetParent;
  end;

  TMateriaPrima = Class(TProducto)
  protected
    function GetValor: Double; override;
    procedure ActualizaTextoNodo; override;
    procedure ActualizaValor; override;
    procedure DoDelete; override;
  End;

  TOperacion = Class(TProducto)
  protected
    function GetValor: Double; override;
    procedure ActualizaTextoNodo; override;
    procedure ActualizaValor; override;
    procedure DoDelete; override;
  End;

  TComponente = Class(TProducto)
  private
    FItems: TList;
    FNotificaciones: TList;
    FSource: TComponente;
    FOwner: IUnknown;
    function GetItems(Index: Integer): TProducto;
    procedure SetItems(Index: Integer; const Value: TProducto);
    procedure SetSource(const Value: TComponente);
    function GetSource: TComponente;
    procedure CopiaContenido(const Value: TComponente);
    procedure SetOwner(const Value: IUnknown);
  protected
    function NuevoNodo: TTreeViewItem;
    function GetValor: Double; override;
    procedure SetPrecio(const Value: Double); override;
    procedure ActualizaTextoNodo; override;
    procedure ActualizaValor; override;
    procedure DoDelete; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AddChildren(AClaseProducto: AClaseBase): Integer;
    function Count: Integer;
    procedure DeleteChildren(AProducto: TProducto);
    property Source: TComponente read GetSource write SetSource;
    property Items[Index: Integer]: TProducto read GetItems write SetItems;
  End;

implementation

uses UComunes;

{ TBase }

constructor TClaseBase.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FNodo:= TTreeViewItem(AOwner);
   FNodo.TagObject:= Self;
end;

destructor TClaseBase.Destroy;
begin
  if FNodo <> Nil then
  begin
    FNodo.Parent:= Nil;
  end;
  inherited Destroy;
end;

function TClaseBase.HasNodo: Boolean;
begin
  Result:= (FNodo <> Nil);
end;

function TClaseBase.GetNodo: TTreeViewItem;
begin
   Result:= FNodo;
end;

{ TProducto }

procedure TProducto.ActualizaTextoNodo;
begin
  Nodo.Text:= Format('ID %d, Nombre  %s,  Cantidad %8.2f, Precio %8.2f', [FID, FNombre, FCantidad, FPrecio]);
end;

constructor TProducto.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FNodo.Text:= 'Producto No Definido';

  FID:= 0;
  FNombre:= '';

  FCantidad:= 0;
  FPrecio:= 0;
  FValor:= 0;

  FExistenCambiosPendientes:= False;
end;

procedure TProducto.Delete;
begin
  DoDelete;
end;


function TProducto.GetCantidad: Double;
begin
   Result:= FCantidad;
end;

function TProducto.GetID: Integer;
begin
   Result:= FID;
end;

function TProducto.GetNombre: String;
begin
   Result:= FNombre;
end;

function TProducto.GetPrecio: Double;
begin
   Result:= FPrecio;
end;

procedure TProducto.SetCantidad(const Value: Double);
begin
   if Value < 0 then
      raise Exception.Create('No se permiten cantidades negativas');
   FCantidad:= Value;
   FExistenCambiosPendientes:= True;
   ActualizaValor;
   ActualizaTextoNodo;
end;

procedure TProducto.SetExistenCambiosPendientes(const Value: Boolean);
begin
  FExistenCambiosPendientes := Value;

  //si existen cambios actualizamos
  if Value then
  begin
    ActualizaValor;
    ActualizaTextoNodo;
  end;

  //establecemos hacia arriba una cadena de notificaciones
  if Value and Assigned(Parent) then Parent.ExistenCambiosPendientes:= True;
end;

procedure TProducto.SetID(const Value: Integer);
begin
   if FID <> Value then
   begin
     FID:= Value;
     ActualizaTextoNodo;
   end;
end;

procedure TProducto.SetNombre(const Value: String);
begin
   if FNombre <> Value then
   begin
     FNombre:= Value;
     ActualizaTextoNodo;
   end;
end;

procedure TProducto.SetParent(const Value: TComponente);
begin
  FParent := Value;
end;

procedure TProducto.SetPrecio(const Value: Double);
begin
   FPrecio:= Value;
   FExistenCambiosPendientes:= True;
   ActualizaValor;
   ActualizaTextoNodo;
end;

{ TMateriaPrima }

procedure TMateriaPrima.ActualizaTextoNodo;
begin
  inherited;
  Nodo.Text:= Format('Materia Prima: ' + Nodo.Text + ', Valor %8.2f ', [FValor]);
end;

procedure TMateriaPrima.ActualizaValor;
begin
   FValor:= Cantidad * Precio;
   ExistenCambiosPendientes:= False;
   //notificamos que se debe revisar el poseedor
   if (Parent <> Nil) then Parent.ExistenCambiosPendientes:= True;
end;

procedure TMateriaPrima.DoDelete;
begin
  if Parent <> nil then
  begin
    Parent.DeleteChildren(Self);
    Parent.ExistenCambiosPendientes:= True;
  end;
end;

function TMateriaPrima.GetValor: Double;
begin
   if ExistenCambiosPendientes then
   begin
     ActualizaValor;
     ActualizaTextoNodo;
   end;
   Result:= FValor;
end;

{ TOperacion }

procedure TOperacion.ActualizaTextoNodo;
begin
  inherited;
  Nodo.Text:= Format('Operacion: ' + Nodo.Text + ', Valor %8.2f ', [FValor]);
end;

procedure TOperacion.ActualizaValor;
begin
   FValor:= Cantidad * Precio;
   ExistenCambiosPendientes:= False;
   //notificamos que se debe revisar el poseedor
   if (Parent <> Nil) then Parent.ExistenCambiosPendientes:= True;
end;

procedure TOperacion.DoDelete;
begin
  with Parent do
  begin
    DeleteChildren(Self);
    ExistenCambiosPendientes:= True;
  end;
end;

function TOperacion.GetValor: Double;
begin
   if ExistenCambiosPendientes then
   begin
     ActualizaValor;
     ActualizaTextoNodo;
   end;
   Result:= FValor;
end;

{ TComponente }

procedure TComponente.ActualizaTextoNodo;
begin
  inherited;
  if Source = Nil then
     Nodo.Text:= Format('Componente: ' + Nodo.Text + #13#10'Valor %8.2f ', [FValor])
  else Nodo.Text:= Format('Componente: ' + Nodo.Text + #13#10'Heredado %8.2f  ValorTotal %8.2f ', [Source.GetValor, FValor]);
end;

procedure TComponente.ActualizaValor;
var
  i: Integer;
begin
  if Source <> nil then FPrecio:= Source.GetValor
  else  FPrecio:= 0;

  for i := 0 to FItems.Count - 1 do
  begin
     FPrecio:= FPrecio +  TProducto(FItems[i]).GetValor;
  end;

  FValor:= FCantidad * FPrecio;

  ExistenCambiosPendientes:= False;
  //notificamos que se debe revisar el poseedor
  if (Parent <> Nil) then
     Parent.ExistenCambiosPendientes:= True;

  for i :=0 to FNotificaciones.Count - 1 do
     TComponente(FNotificaciones[i]).ExistenCambiosPendientes:= True;
end;

function TComponente.AddChildren(AClaseProducto: AClaseBase): Integer;
begin
  if Assigned(AClaseProducto) then
  begin
    Result:= FItems.Add(AClaseProducto.Create(NuevoNodo));
    TProducto(FItems[Result]).Parent:= Self;
  end
  else raise Exception.Create('Referencia a objeto no valida');
end;

procedure TComponente.CopiaContenido(const Value: TComponente);
begin
   //podemos optar por copiar en el componente actual el contenido
   //en lugar de asignar que se acople uno en el otro
   Value.ActualizaValor;
   Value.ActualizaTextoNodo;
   Value.Parent:= Self;
   //acciones de copia
   //-als que  sean
      //ID:= Value.ID;
      //Nombre:= Value.Nombre;
      //Cantidad:= 1;
end;

function TComponente.Count: Integer;
begin
   Result:= FItems.Count;
end;

constructor TComponente.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FItems:= TList.Create;
  FNotificaciones:= TList.Create;
end;

procedure TComponente.DeleteChildren(AProducto: TProducto);
var
  i: Integer;
begin
   i:= FItems.IndexOf(AProducto);
   if i > -1 then
   begin
     TProducto(FItems[i]).Nodo.Free;
     FItems.Delete(i);
    end;
end;

destructor TComponente.Destroy;
begin
  FItems.Clear;
  FItems.Free;
  FItems:= Nil;

  FNotificaciones.Clear;
  FNotificaciones.Free;
  FNotificaciones:= Nil;

  inherited Destroy;
end;

// pte revisar borrado de un componente
//acoplado a otro componente
procedure TComponente.DoDelete;
begin
  if Parent <> nil then
  begin
    //Parent.DeleteChildren(Self);
    Source:= Nil;
    Parent.ExistenCambiosPendientes:= True;
    Nodo.Free;
  end
  else Nodo.Free;
end;

function TComponente.GetItems(Index: Integer): TProducto;
begin
  RequerirRangoValores(0, FItems.Count-1, Index);
  Result:= TProducto(FItems[Index]);
end;

function TComponente.GetSource: TComponente;
begin
  Result:= FSource;
end;

function TComponente.GetValor: Double;
begin
  if ExistenCambiosPendientes then
  begin
    ActualizaValor;
    ActualizaTextoNodo;
  end;
  Result:= FValor;
end;

function TComponente.NuevoNodo: TTreeViewItem;
begin
  Result:= TTreeViewItem.Create(FNodo);
  Result.Font.Size:= 14;
  Result.Parent:= FNodo;
end;

procedure TComponente.SetItems(Index: Integer; const Value: TProducto);
begin
  RequerirRangoValores(0, FItems.Count-1, Index);
  FItems[Index]:= Value;
end;

procedure TComponente.SetOwner(const Value: IUnknown);
begin
  FOwner := Value;
end;

procedure TComponente.SetPrecio(const Value: Double);
begin
  raise Exception.Create('Propiedad de solo lectura');
end;

procedure TComponente.SetSource(const Value: TComponente);
var
  i: Integer;
  btns: TMsgDlgButtons;
begin
  if Value <> FSource then
  begin
    if FSource <> nil then
    begin
      if Value <> nil then
      begin
        Include(btns, System.UITypes.TMsgDlgBtn.mbYes);
        Include(btns, System.UITypes.TMsgDlgBtn.mbNo);
        if MessageDlg('Si sigues adelante eliminars el actual contenido componente '+ FSource.Nodo.Text,
                      System.UITypes.TMsgDlgType.mtWarning,
                      btns,
                      0,
                      System.UITypes.TMsgDlgBtn.mbNo
                      ) <> mrYes then Exit;
      end;


      i:= FNotificaciones.IndexOf(FSource);
      if i >= 0 then FNotificaciones.Delete(i);
      FSource.Nodo.Free;
    end;


    FSource := Value;

    if FSource <> nil then
    begin
       CopiaContenido(Value);
       if Value.FNotificaciones.IndexOf(Self) = -1 then
          Value.FNotificaciones.Add(Self);
    end;

    ActualizaValor;
    ActualizaTextoNodo;
  end;
end;

end.
