unit USJTreeView;

interface

uses
  System.SysUtils, System.Classes, FMX.Types, FMX.Layouts, FMX.TreeView;

type
  TNotifyOnSelectEvent = procedure(Sender: TObject; ABase: TClass)of object;

  THelperTreeViewItem = class helper for TTreeViewItem
  public
    //Navegacion relativa a hijos de Nodo
    function First: TTreeViewItem;
    function Last:  TTreeViewItem;
    function Next:  TTreeViewItem;
    function Prior: TTreeViewItem;
    function Bof: Boolean;
    function Eof: Boolean;
    //Navegacion global
    function FirstGlobal: TTreeViewItem;
    function LastGlobal:  TTreeViewItem;
    function NextGlobal:  TTreeViewItem;
    function PriorGlobal: TTreeViewItem;
    function BofGlobal: Boolean;
    function EofGlobal: Boolean;
  end;

  TSJTreeView = class(TTreeView)
  private
    { Private declarations }
    FOnSelectEvent: TNotifyOnSelectEvent;
    procedure SetOnSelectEvent(const Value: TNotifyOnSelectEvent);
  protected
    { Protected declarations }
    function NuevoNodo: TTreeViewItem;
    procedure DoSelectEvent(const Value: TTreeViewItem); virtual;
  public
    { Public declarations }
    procedure SetSelected(const Value: TTreeViewItem); override;
  published
    { Published declarations }
    property OnSelectEvent: TNotifyOnSelectEvent read FOnSelectEvent write SetOnSelectEvent;
  end;

  TSJTreeViewItem = class(TTreeViewItem)
    procedure ApplyStyle; override;
  end;



procedure Register;

implementation


uses FMX.Objects;

procedure Register;
begin
  RegisterComponents('Samples', [TSJTreeView]);
end;

{ TSJTreeView }

procedure TSJTreeView.DoSelectEvent(const Value: TTreeViewItem);
begin
   if Assigned(OnSelectEvent) and (Value.TagObject <> Nil) then
      OnSelectEvent(Self, (Value.TagObject).ClassType);
end;

function TSJTreeView.NuevoNodo: TTreeViewItem;
begin
  Result:= TTreeViewItem.Create(Self);
  Result.Parent:= Self;
end;

procedure TSJTreeView.SetOnSelectEvent(const Value: TNotifyOnSelectEvent);
begin
  FOnSelectEvent := Value;
end;

procedure TSJTreeView.SetSelected(const Value: TTreeViewItem);
begin
   inherited SetSelected(Value);
   DoSelectEvent(Value);
end;

{ THelperTreeViewItem }

function THelperTreeViewItem.Bof: Boolean;
begin
   Result:= (0 = Index);
end;

function THelperTreeViewItem.BofGlobal: Boolean;
begin
   Result:= (0 = GlobalIndex);
end;

function THelperTreeViewItem.Eof: Boolean;
begin
   if ParentItem = Nil then
   begin
     Result:= (TreeView.Count - 1 = Index);
   end
   else begin
     Result:= (ParentItem.Count - 1 = Index);
   end;
end;

function THelperTreeViewItem.EofGlobal: Boolean;
begin
   Result:= (TreeView.GlobalCount - 1 = GlobalIndex);
end;

function THelperTreeViewItem.First: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;
begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    FNode.IsSelected:= False;

    if ParentItem = Nil then
    begin
      Result:= TreeView.ItemByIndex(0);
    end
    else begin
      Result:= ParentItem.ItemByIndex(0);
    end;
    Result.IsSelected:= True;
    TreeView.Selected:= Result;
  end;
end;

function THelperTreeViewItem.FirstGlobal: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;
begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    FNode.IsSelected:= False;
    Result:= TreeView.ItemByGlobalIndex(0);
    Result.IsSelected:= True;
    TreeView.Selected:= Result;
  end;
end;

function THelperTreeViewItem.Last: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;
  FCount: Integer;
begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    FNode.IsSelected:= False;

    if ParentItem = Nil then
    begin
      FCount:= TreeView.Count;
      Result:= TreeView.ItemByIndex(FCount-1);
    end
    else begin
      FCount:= ParentItem.Count;
      Result:= ParentItem.ItemByIndex(FCount-1);
    end;
    Result.IsSelected:= True;
    TreeView.Selected:= Result;
  end;
end;

function THelperTreeViewItem.LastGlobal: TTreeViewItem;
var
  FNode: TTreeViewItem;
begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    FNode.IsSelected:= False;
    Result:= TreeView.ItemByGlobalIndex(TreeView.GlobalCount-1);
    Result.IsSelected:= True;
    TreeView.Selected:= Result;
  end;
end;

function THelperTreeViewItem.Next: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;
  FCount: Integer;

begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    Idx:= FNode.Index;

    if ParentItem = Nil then FCount:= TreeView.Count
    else FCount:= ParentItem.Count;

    if Idx < FCount - 1 then
    begin
       FNode.IsSelected:= False;
       Inc(Idx);

       if ParentItem = Nil then Result:= TreeView.ItemByIndex(Idx)
       else Result:= ParentItem.ItemByIndex(Idx);
       Result.IsSelected:= True;
       TreeView.Selected:= Result;
    end;
  end;
end;

function THelperTreeViewItem.NextGlobal: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;

begin
  Result:= Nil;
  FNode:= Self;
  if (FNode <> Nil) then
  begin
    Idx:= FNode.GlobalIndex;

    if Idx < TreeView.GlobalCount - 1 then
    begin
       FNode.IsSelected:= False;
       Inc(Idx);
       Result:= TreeView.ItemByGlobalIndex(Idx);
       Result.IsSelected:= True;
       TreeView.Selected:= Result;
    end;
  end;
end;

function THelperTreeViewItem.Prior: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;

begin
  Result:= Nil;

  FNode:= Self;
  if FNode <> Nil then
  begin
    Idx:= FNode.Index;

    if Idx > 0 then
    begin
       FNode.IsSelected:= False;
       Dec(Idx);
       if ParentItem = Nil then Result:= TreeView.ItemByIndex(Idx)
       else Result:= ParentItem.ItemByIndex(Idx);
       Result.IsSelected:= True;
       TreeView.Selected:= Result;
    end;
  end;
end;

function THelperTreeViewItem.PriorGlobal: TTreeViewItem;
var
  FNode: TTreeViewItem;
  Idx: Integer;

begin
  Result:= Nil;

  FNode:= Self;
  if FNode <> Nil then
  begin
    Idx:= FNode.GlobalIndex;

    if Idx > 0 then
    begin
       FNode.IsSelected:= False;
       Dec(Idx);
       Result:= TreeView.ItemByGlobalIndex(Idx);
       Result.IsSelected:= True;
       TreeView.Selected:= Result;
    end;
  end;
end;

{ TSJTreeViewItem }

procedure TSJTreeViewItem.ApplyStyle;
var
  img: TFmxObject;
begin
  inherited;
  img:= FindStyleResource('Image');
  if img <> nil then
  begin
    if img is TImage then
       (img as TImage).Bitmap.LoadFromFile('NodoProducto24x24.gif');
  end;
end;

end.
