unit UCalculadora;

interface

uses System.SysUtils, System.Variants, System.Classes, Generics.Collections,
  UCBClasses;

const
  MaxDigitos = 12;
type
  TCalculadoraBasica = class(TComponent)
  private
    FDisplay: String;
    FlagNumero: Boolean;
    Operando: Double;
    Operador: String;
    FCatalogo: TObjectDictionary<String, TDigitoOperador>;
    procedure DesplazarDigitoALaIzquierda(AOperando: Char);
    procedure IntroduceDigito(AOperando: Char);
    function EsOperadorInmediato(const ALexema: String): Boolean;
    function GetOperadorText: String;
  protected
    procedure ActualizaDisplay(const AOperando: String); virtual;
    procedure DoReset; virtual;
    procedure RegisterCatalogo(ACatalogo: Array of TEtiqueta); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ExisteError: Boolean;
    function LeeDisplay: String;
    function ProcesarDigito(const ALexema: Char): String;
    function ProcesarOperacion(const ALexema: String): String;
    procedure RegisterEtiqueta(AEtiqueta: TEtiqueta);
    function Reset: String;
    property OperadorText: String read GetOperadorText;
  end;

implementation

uses Rtti;

{ TCalculadoraBasica }

procedure TCalculadoraBasica.ActualizaDisplay(const AOperando: String);
begin
  FDisplay:= AOperando;
end;

constructor TCalculadoraBasica.Create(AOwner: TComponent);
begin
  inherited;
  FCatalogo:= TObjectDictionary<String, TDigitoOperador>.Create;
  RegisterCatalogo(CatalogoSimbolosBasicos);
  Operando:= 0;
  Operador:= ' ';
  FlagNumero:= False;
end;

procedure TCalculadoraBasica.DesplazarDigitoALaIzquierda(AOperando: Char);
begin
  FDisplay:= FDisplay + AOperando;
end;

destructor TCalculadoraBasica.Destroy;
var
  KeyLexema: String;
begin
  for KeyLexema in FCatalogo.Keys do FCatalogo.Items[KeyLexema].Free;
  FCatalogo.Clear;
  FreeAndNil(FCatalogo);
  inherited Destroy;
end;

function TCalculadoraBasica.EsOperadorInmediato(const ALexema: String): Boolean;
begin
  if (ALexema = ' ') or (ALexema = '=') then
    Result:= False
  else Result:= FCatalogo.Items[ALexema].EsOperadorInmediato;
end;

function TCalculadoraBasica.ExisteError: Boolean;
begin
  Result:= False;
end;

function TCalculadoraBasica.GetOperadorText: String;
begin
  if (Operador = ' ') or (Operador = '=') then
    Result:= Operador
  else Result:= FCatalogo.Items[Operador].GetRepresentacion;
end;

procedure TCalculadoraBasica.IntroduceDigito(AOperando: Char);
begin
  if AOperando = ',' then
    FDisplay:= '0' + AOperando
  else
    FDisplay:= AOperando;

  if FDisplay <> '0' then FlagNumero:= True;
end;

function TCalculadoraBasica.LeeDisplay: String;
begin
  Result:= FDisplay
end;

function TCalculadoraBasica.ProcesarDigito(const ALexema: Char): String;
begin
   if FlagNumero then
    begin
      if Length(FDisplay) < MaxDigitos then
      begin
         if (FDisplay = '0') and (ALexema = '0') then
           Exit(LeeDisplay)
         else
          begin
            //evaluamos si ya exite el punto decimal
            if (ALexema = ',') and (StrScan(Pchar(FDisplay), ',') <> nil) then
              Exit(LeeDisplay);

            DesplazarDigitoALaIzquierda(ALexema);
          end;
      end;
    end
   else
    IntroduceDigito(ALexema);

   Result:= LeeDisplay;
end;

function TCalculadoraBasica.ProcesarOperacion(const ALexema: String): String;
begin
   if EsOperadorInmediato(ALexema) then Operador:= ALexema;

   if ((Operador = ' ') or (Operador = '=')) then
     Operando:= StrToFloat(FDisplay)
   else
     Operando:= FCatalogo.Items[Operador].ExecuteAction(Self, Operando, StrToFloat(FDisplay));

   ActualizaDisplay(FloatToStr(Operando));
   FlagNumero:= False;
   Operador:= ALexema;

   Result:= LeeDisplay;
end;

procedure TCalculadoraBasica.RegisterCatalogo(ACatalogo: array of TEtiqueta);
var
  i: Integer;
  FClass: TDigitoClass;
begin
  for i:= Low(ACatalogo) to High(ACatalogo) do
  begin
    FClass:= ACatalogo[i].Clase;
    FCatalogo.Add(ACatalogo[i].Lexema, FClass.Create(ACatalogo[i], self));
  end;
end;


procedure TCalculadoraBasica.RegisterEtiqueta(AEtiqueta: TEtiqueta);
var
  FClass: TDigitoClass;
begin
  FClass:= AEtiqueta.Clase;
  FCatalogo.Add(AEtiqueta.Lexema, FClass.Create(AEtiqueta, self));
end;

function TCalculadoraBasica.Reset: String;
begin
  DoReset;
  Result:= LeeDisplay;
end;

procedure TCalculadoraBasica.DoReset;
begin
   if FDisplay = '0' then
    begin
      Operando:= 0;
      Operador:= ' ';
    end
   else
      FDisplay:= '0';
   FlagNumero:= False;
end;

end.
