unit uReloj;

interface

uses
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants,
  FMX.Types, FMX.Graphics, FMX.Controls, FMX.Forms, FMX.Dialogs, FMX.StdCtrls,
  System.IOUtils, Generics.Collections, System.StrUtils;

const
   NOMFILEMOV = 'MOVIMIENTOS.DAT';
   NOMFILEUSER = 'USER.DAT';
   SEPARADOR = '#';
   INTERVAL = 20000;

   { cadenas error }
   ERROR_IDENTIFICACION = 'Error autentificacion';

type
  TInfoRegistro = record
    Index: Integer;
    IDEmpleado: Integer;
    Hora: TDateTime;
    IDTipo: Integer;
  end;

  TEmpleado = class(TObject)
  private
    FApellido1: string;
    FApellido2: string;
    FEnProceso: Boolean;
    FId_Empleado: Integer;
    FNombre: string;
    FOnBeforeAnotacion: TNotifyEvent;
    FOnRelease: TNotifyEvent;
    FTimer: TTimer;
    FTarjeta: String;
    FPassword: string;
    FTimeToFinish: TDateTime;
    procedure DoTiempoExcedido(Sender: TObject);
    procedure SetApellido1(const Value: string);
    procedure SetApellido2(const Value: string);
    procedure SetNombre(const Value: string);
    procedure SetTarjeta(const Value: String);
    procedure SetPassword(const Value: string);
 protected
    constructor CreateInstance;
    class function AccessInstance(Request: Integer): TEmpleado;
    procedure DoBeforeAnotacion; dynamic;
    procedure DoRelease; dynamic;
    function IdentificarEmpleado: Boolean; virtual;
    procedure LoadData(const AFileName: String); virtual;
  public
    constructor Create;
    destructor Destroy; override;
    function AutentificarEmpleado(const APassword: String): Boolean;
    function GetNombreCompleto: string;
    class function Instance: TEmpleado;
    procedure SetIdentidad; virtual;
    procedure PrintInforme(const AFileName: TFileName);
    function GetRegistrosHorarios: TList<TInfoRegistro>;
    procedure RegistrarMovimiento(const ATipo: Integer); virtual;
    procedure CancelarMovimiento;
    class procedure GuardarMovimientos;
    procedure MarcarEnviado(AIndex: Integer); virtual;
    class procedure DeleteMovimientos; virtual;
    class procedure ReleaseInstance;
    property Apellido1: string read FApellido1 write SetApellido1;
    property Apellido2: string read FApellido2 write SetApellido2;
    property EnProceso: Boolean read FEnProceso default False;
    property Id_Empleado: Integer read FId_Empleado;
    property Tarjeta: String read FTarjeta write SetTarjeta;
    property Nombre: string read FNombre write SetNombre;
    property Password: string read FPassword write SetPassword;
    property OnBeforeAnotacion: TNotifyEvent read FOnBeforeAnotacion write
            FOnBeforeAnotacion;
    property OnRelease: TNotifyEvent read FOnRelease write FOnRelease;
    property TimeToFinish: TDateTime read FTimeToFinish;
  end;


implementation

uses IniFiles, DateUtils;

type
  TMovimientos = class;

  TMovimiento = class(TComponent)
  private
    FEmpleado: Integer;
    FHora: TDateTime;
    FIncidencia: Integer;
    FLista: TMovimientos;
    FIDTipo: Integer;
    FDeleted: Boolean;
    procedure SetEmpleado(Value: Integer);
    procedure SetHora(Value: TDateTime);
    procedure SetIncidencia(Value: Integer);
    procedure SetIDTipo(const Value: Integer);
    procedure SetDeleted(const Value: Boolean);
  public
    constructor Create(AOwner: TComponent); override;
    constructor CreateInstance(AOwner: TComponent; ALista: TMovimientos);
    destructor Destroy; override;
    function GetCadenaRegistro: string; virtual;
    class function LoadObject(const ACadena: String; ALista: TMovimientos):
            TMovimiento; virtual;
  published
    property Empleado: Integer read FEmpleado write SetEmpleado;
    property Hora: TDateTime read FHora write SetHora;
    property Incidencia: Integer read FIncidencia write SetIncidencia;
    property IDTipo: Integer read FIDTipo write SetIDTipo;
    property Deleted: Boolean read FDeleted write SetDeleted;
  end;

  TIteradorMovimientos = class(TObject)
  private
    FIndex: Integer;
    FListaMovimientos: TMovimientos;
    function GetCurrent: TMovimiento;
    function GetFirst: TMovimiento;
    function GetLast: TMovimiento;
    function GetNext: TMovimiento;
    function GetPrevious: TMovimiento;
  public
    constructor Create(ALista: TMovimientos);
    destructor Destroy; override;
    property Current: TMovimiento read GetCurrent;
    property First: TMovimiento read GetFirst;
    property Last: TMovimiento read GetLast;
    property Next: TMovimiento read GetNext;
    property Previous: TMovimiento read GetPrevious;
  end;

  TMovimientos = class(TObject)
  private
    FInternal: TStrings;
    FModified: Boolean;
    function GetCadenas(Index: Integer): string;
    function GetCount: Integer;
    function GetItems(Index: integer): TMovimiento;
    procedure LoadFromFile(const AFilename: TFilename); overload;
    procedure SaveToFile(const AFilename: TFilename); overload;
    procedure SetCadenas(Index: Integer; const Value: string);
    procedure SetItems(Index: integer; Value: TMovimiento);
  public
    constructor Create;
    destructor Destroy; override;
    function Add(AMovimiento: TMovimiento): Integer; virtual;
    function CreateIterator: TIteradorMovimientos;
    procedure Delete(const AIndex: Integer);
    procedure LoadFromFile; overload;
    procedure Modified;
    procedure Clear;
    procedure Print(const AFilename: TFileName);
    function GetRegistrosHorarios: TList<TInfoRegistro>;
    procedure RegistrarMovimiento(const AEmpleado: Integer; const ADiaHora:
            TDateTime; const AIDTipo: Integer; AMustSave: Boolean);
    procedure ReleaseIterator(var it: TIteradorMovimientos);
    procedure SaveToFile; overload;
    property Cadenas[Index: Integer]: string read GetCadenas write SetCadenas;
    property Count: Integer read GetCount;
    property Items[Index: integer]: TMovimiento read GetItems write SetItems;
            default;
  end;

var
  Lista: TMovimientos;


{
********************************** TEmpleado ***********************************
}
procedure TEmpleado.CancelarMovimiento;
begin
   ReleaseInstance;
end;

constructor TEmpleado.Create;
begin
  inherited Create;
  raise Exception.CreateFmt('Access class %s through Instance only',
          [ClassName]);
end;

constructor TEmpleado.CreateInstance;
begin
  inherited Create;
  FId_Empleado:= -1;
  FNombre:= '';
  FApellido1:= '';
  FApellido2:= '';
  FEnProceso:= False;
  FTimer:= TTimer.Create(nil);
  FTimer.Enabled:= False;
  FTimer.Interval:= INTERVAL; //segundos antes de eliminar usuario indeciso
  FTimer.OnTimer:= DoTiempoExcedido;
end;

procedure TEmpleado.MarcarEnviado(AIndex: Integer);
begin
  if FId_Empleado = -1 then
     Raise Exception.Create('Error: Falta proceso de identificacion usuario');
  //anota el movimiento
  if Assigned(Lista) and (AIndex >= 0) then Lista[AIndex].Deleted:= True;
end;

class procedure TEmpleado.DeleteMovimientos;
var
  i: Integer;
begin
  if Assigned(Lista) then
  begin
    for i := Lista.Count-1 downto 0 do Lista.Delete(i);
    Lista.SaveToFile;
  end;
end;

destructor TEmpleado.Destroy;
begin
  FEnProceso:= False;
  FTimer.OnTimer:= nil;
  FreeAndNil(FTimer);
  DoRelease;
  if AccessInstance(0) = Self then AccessInstance(2);
  inherited Destroy;
end;

class function TEmpleado.AccessInstance(Request: Integer): TEmpleado;

  {$WRITEABLECONST ON}
  const FInstance: TEmpleado = nil;
  {$WRITEABLECONST OFF}

begin
  case Request of
    0 : ;
    1 : if not Assigned(FInstance) then FInstance := CreateInstance;
    2 : FInstance := nil;
  else
    raise Exception.CreateFmt('Illegal request %d in AccessInstance',
            [Request]);
  end;
  Result := FInstance;
end;

function TEmpleado.AutentificarEmpleado(const APassword: String): Boolean;
begin
  Result:= (CompareStr(APassword, FPassword) = 0);
end;

procedure TEmpleado.DoBeforeAnotacion;
begin
  if Assigned(FOnBeforeAnotacion) then FOnBeforeAnotacion(Self);
end;

procedure TEmpleado.DoRelease;
begin
  if Assigned(FOnRelease) then FOnRelease(Self);
end;

procedure TEmpleado.DoTiempoExcedido(Sender: TObject);
begin
  ReleaseInstance;
end;

function TEmpleado.GetNombreCompleto: string;
begin
  Result:= FNombre + ' ' + FApellido1 + ' ' + FApellido2;
end;

function TEmpleado.GetRegistrosHorarios: TList<TInfoRegistro>;
begin
  if Assigned(Lista) then
    Result:= Lista.GetRegistrosHorarios
  else Result:= Nil;
end;

class procedure TEmpleado.GuardarMovimientos;
begin
  if Assigned(Lista) then
  begin
    Lista.SaveToFile;
  end;
end;

function TEmpleado.IdentificarEmpleado: Boolean;
begin
  Result:= False;
  LoadData(NOMFILEUSER);
  Result:= FID_Empleado > 0
end;

class function TEmpleado.Instance: TEmpleado;
begin
  Result := AccessInstance(1);
end;

procedure TEmpleado.LoadData(const AFileName: String);
var
  FFileName: string;
begin
  //'/data/data/com.embarcadero.HeaderFooterApplication/files/USER.DAT'
  FFileName:= TPath.Combine(TPath.GetHomePath, AFileName);
  if FileExists(FFileName) then begin
    with TIniFile.Create(FFileName)do begin
      try
         //escritura de las propiedades
         FId_Empleado:= ReadInteger('DATOS', 'ID', -1);
         FNombre:= ReadString('DATOS', 'NOMBRE', '');
         FApellido1:= ReadString('DATOS', 'APELLIDOS1', '');
         FApellido2:= ReadString('DATOS', 'APELLIDOS2', '');
         FTarjeta:= ReadString('DATOS', 'TARJETA', '');
         FPassword:= ReadString('DATOS', 'PASSWORD', '');
      finally
         Free;
      end;
    end;
  end;
end;

procedure TEmpleado.SetIdentidad;
var
  OkUser: Boolean;
  tmpPassword: string;
  i, ii: Integer;
begin
  i:= 0;
  FEnProceso:= True;

  //verificamos que existe el fichero de identificacion
  OkUser:= IdentificarEmpleado;
  if OkUser then
  begin
     //seguimos adelante dejando al usuario enlazar un evento previo a la anotacion
     DoBeforeAnotacion;
     FTimeToFinish:= IncSecond(Now, INTERVAL DIV 1000);
     FTimer.Enabled:= True;
  end
  else begin
     ReleaseInstance;
     Raise Exception.Create(ERROR_IDENTIFICACION);
  end;
end;

procedure TEmpleado.PrintInforme(const AFileName: TFileName);
begin
  if Assigned(Lista) then Lista.Print(AFilename);
end;

procedure TEmpleado.RegistrarMovimiento(const ATipo: Integer);
begin
  if FId_Empleado = -1 then
     Raise Exception.Create('Error: Falta proceso de identificacion usuario');
  //anota el movimiento
  if Assigned(Lista) then Lista.RegistrarMovimiento(FId_Empleado, Now, ATipo, False);
  ReleaseInstance;
end;

class procedure TEmpleado.ReleaseInstance;
var
  FEmpleado: TEmpleado;
begin
  {$IF DEFINED(iOS) or DEFINED(ANDROID)}
    FEmpleado:= AccessInstance(0);
    FEmpleado.DoRelease;
    FEmpleado.Free;
  {$ELSE}
    AccessInstance(0).Free;
  {$ENDIF}
end;

procedure TEmpleado.SetApellido1(const Value: string);
begin
  if Value <> FApellido1 then begin
     FApellido1:= Value;
  end;
end;

procedure TEmpleado.SetApellido2(const Value: string);
begin
  if Value <> FApellido2 then begin
     FApellido2:= Value;
  end;
end;

procedure TEmpleado.SetNombre(const Value: string);
begin
  if Value <> FNombre then begin
     FNombre:= Value;
  end;
end;

procedure TEmpleado.SetPassword(const Value: string);
begin
  FPassword := Value;
end;

procedure TEmpleado.SetTarjeta(const Value: String);
begin
  FTarjeta := Value;
end;

{
********************************* TMovimiento **********************************
}
constructor TMovimiento.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

constructor TMovimiento.CreateInstance(AOwner: TComponent; ALista:
        TMovimientos);
begin
  inherited Create(AOwner);
  FLista:= ALista;
end;

destructor TMovimiento.Destroy;
begin
  FLista:= Nil;
  inherited Destroy;
end;

function TMovimiento.GetCadenaRegistro: string;

  function ComponentToString(Component: TComponent): string;
  var
    BinStream:TMemoryStream;
    StrStream: TStringStream;
    s: string;
    i: Integer;
  begin
    s:= '';
    i:= 0;
    BinStream := TMemoryStream.Create;
    try
      StrStream := TStringStream.Create(s);
      try
        BinStream.WriteComponent(Component);
        BinStream.Seek(0, soFromBeginning);
        ObjectBinaryToText(BinStream, StrStream);
        StrStream.Seek(0, soFromBeginning);
        s:= StrStream.DataString;

       {$IF DEFINED(iOS) or DEFINED(ANDROID)}
          i:= Pos(#10, s);
          while i>0 do begin
             s:= copy(s, Low(s), i-1) + SEPARADOR + copy(s, i+2, length(s)-i);
             i:= Pos(#10, s);
          end;
          s:= ReplaceStr(s, '#nd#', '#end#');
       {$ELSE}
          i:= Pos(#13#10, s);
          while i>0 do begin
             s:= copy(s, Low(s), i-1) + SEPARADOR + copy(s, i+2, length(s)-i);
             i:= Pos(#13#10, s);
          end;
       {$ENDIF}
        Result:= s;
      finally
        StrStream.Free;
      end;
    finally
      BinStream.Free
    end;
  end;

begin
  Result:= ComponentToString(self);
end;

class function TMovimiento.LoadObject(const ACadena: String; ALista:
        TMovimientos): TMovimiento;

        function StringToComponent(Value: string): TComponent;
        var
          StrStream:TStringStream;
          BinStream: TMemoryStream;
          s: String;
          i: Integer;
        begin
          s:= Value;
          i:= Pos(SEPARADOR, s);
          while i>0 do begin
             {$IF DEFINED(iOS) or DEFINED(ANDROID)}
                 s:= copy(s, Low(s), i-1) + #10 + copy(s, i+1, length(s)-i);
             {$ELSE}
                 s:= copy(s, Low(s), i-1) + #13#10 + copy(s, i+1, length(s)-i);
             {$ENDIF}
             i:= Pos('#', s);
          end;
          StrStream := TStringStream.Create(s);
          try
            BinStream := TMemoryStream.Create;
            try
              ObjectTextToBinary(StrStream, BinStream);
              BinStream.Seek(0, soFromBeginning);
              Result := BinStream.ReadComponent(nil);
            finally
              BinStream.Free;
            end;
          finally
            StrStream.Free;
          end;
        end;
  var
    c: TComponent;

begin
  if ACadena = '' then Exit(Nil);

  c:= StringToComponent(ACadena);
  if (c is TMovimiento) then begin
     TMovimiento(c).FLista:= ALista;
     Result:= (c as TMovimiento);
  end
  else Result:= nil;
end;

procedure TMovimiento.SetDeleted(const Value: Boolean);
begin
  if Value <> FDeleted then FDeleted := Value;
end;

procedure TMovimiento.SetEmpleado(Value: Integer);
begin
  if Value <> FEmpleado then FEmpleado:= Value;
end;

procedure TMovimiento.SetHora(Value: TDateTime);
begin
  if Value <> FHora then FHora:= Value;
end;

procedure TMovimiento.SetIDTipo(const Value: Integer);
begin
  if Value <> FIDTipo then FIDTipo := Value;
end;

procedure TMovimiento.SetIncidencia(Value: Integer);
begin
  if Value <> FIncidencia then FIncidencia:= Value;
end;

{
***************************** TIteradorMovimientos *****************************
}
constructor TIteradorMovimientos.Create(ALista: TMovimientos);
begin
  inherited Create;
  FListaMovimientos := ALista;
  FIndex      := -1;
end;

destructor TIteradorMovimientos.Destroy;
begin
  FListaMovimientos:= nil;
  inherited Destroy;
end;

function TIteradorMovimientos.GetCurrent: TMovimiento;
begin
  if FIndex = -1 then
     result := nil
  else
     result := FListaMovimientos[FIndex];
end;

function TIteradorMovimientos.GetFirst: TMovimiento;
begin
  FIndex := 0;
  result := GetCurrent;
end;

function TIteradorMovimientos.GetLast: TMovimiento;
begin
  FIndex := FListaMovimientos.count - 1;
  result := GetCurrent;
end;

function TIteradorMovimientos.GetNext: TMovimiento;
begin
  Inc(FIndex);
  if (FIndex >= FListaMovimientos.count) then
     FIndex   := -1;

  result := GetCurrent;
end;

function TIteradorMovimientos.GetPrevious: TMovimiento;
begin
  Dec(FIndex);
  result := GetCurrent;
end;

{
********************************* TMovimientos *********************************
}
procedure TMovimientos.Clear;
begin
  while Count > 0 do begin
     TMovimiento(Items[0]).Free;
     Items[0]:= Nil;
     Delete(0);
  end;
end;

constructor TMovimientos.Create;
var
  FFileHandle: Integer;
  FFileName: String;
begin
  FInternal:= TStringList.Create;
  //si no existiera el fichero lo crearemos
  FFilename:= TPath.Combine(TPath.GetHomePath, NOMFILEMOV);

  //DeleteFile(FFilename);
  if not FileExists(FFilename) then begin
     FFileHandle:= FileCreate(FFilename);
     FileClose(FFileHandle);
  end
  else  //cargamos el fichero en memoria
     LoadFromFile;
end;

destructor TMovimientos.Destroy;
var
  FFileName: String;
begin
  FFilename:= TPath.Combine(TPath.GetHomePath, NOMFILEMOV);
  if FModified then SaveToFile(FFilename);
  Clear;
  FreeAndNil(FInternal);
  inherited Destroy;
end;

function TMovimientos.Add(AMovimiento: TMovimiento): Integer;
begin
  Result:= -1;
  if AMovimiento <> nil then begin
     Result:= FInternal.AddObject(AMovimiento.GetCadenaRegistro, AMovimiento);
     FModified:= True;
  end;
end;

function TMovimientos.CreateIterator: TIteradorMovimientos;
begin
  result := TIteradorMovimientos.Create(self);
end;

procedure TMovimientos.Delete(const AIndex: Integer);
begin
  FInternal.Delete(AIndex);
  FModified:= True;
end;

function TMovimientos.GetCadenas(Index: Integer): string;
begin
  if (Index < 0) or (Index >= FInternal.Count) then
     Raise Exception.Create('Error indice fuera de rango');
  Result:= FInternal[Index];
end;

function TMovimientos.GetCount: Integer;
begin
  Result:= FInternal.Count;
end;

function TMovimientos.GetItems(Index: integer): TMovimiento;
begin
  if (Index < 0) or (Index >= FInternal.Count) then
     Raise Exception.Create('Error indice fuera de rango');
  Result:= TMovimiento(FInternal.Objects[Index]);
end;

function TMovimientos.GetRegistrosHorarios: TList<TInfoRegistro>;
var
  fList: TList<TInfoRegistro>;
  fInfoRegistro: TInfoRegistro;
  it: TIteradorMovimientos;
  Mov: TMovimiento;
  l: TStrings;
begin
  Result:= Nil;
  fList:= TList<TInfoRegistro>.Create;
  l:= TStringList.Create;
  // se utiliza el iterador para acceder al primer elemto.
  it := CreateIterator();
  try
     // cancelar los hilos y esperar a que cada uno de ellos se haya cancelado
     while it.Next <> nil do
     begin
        Mov := it.Current;
        if not Mov.Deleted then
        begin
          fInfoRegistro.Index:= it.FIndex;
          fInfoRegistro.IDEmpleado:= Mov.Empleado;
          fInfoRegistro.Hora:= Mov.Hora;
          fInfoRegistro.IDTipo:= Mov.IDTipo;
          fList.Add(fInfoRegistro);
        end;
     end;
     Result:= fList;
  finally
     ReleaseIterator(it);
     l.Free;
  end;
end;

procedure TMovimientos.LoadFromFile;
var
  FFileName: String;
begin
  FFileName:= TPath.Combine(TPath.GetHomePath, NOMFILEMOV);
  LoadFromFile(FFileName);
end;

procedure TMovimientos.LoadFromFile(const AFilename: TFilename);
var
  i: Integer;
  s: String;
begin
  if Assigned(FInternal) then begin
     FInternal.LoadFromFile(AFilename);
     s:= FInternal.Text;
     if s <> #13#10 then
     begin
       for i:= Count - 1 downto 0 do
       begin
         Items[i]:= TMovimiento.LoadObject(Cadenas[i], Self);
         if Items[i] = Nil then FInternal.Delete(i);
       end;
     end;
  end;
end;

procedure TMovimientos.Modified;
begin
  FModified:= True;
end;

procedure TMovimientos.Print(const AFilename: TFileName);
var
  it: TIteradorMovimientos;
  Mov: TMovimiento;
  l: TStrings;
begin
  l:= TStringList.Create;
  // se utiliza el iterador para acceder al primer elemto.
  it := CreateIterator();
  try
     // cancelar los hilos y esperar a que cada uno de ellos se haya cancelado
     while it.Next <> nil do
     begin
        Mov := it.Current;
        l.Add(Mov.GetCadenaRegistro);
     end;
     if l.Count > 0 then begin
        l.SaveToFile(AFilename);
     end;
  finally
     ReleaseIterator(it);
     l.Free;
  end;
end;

procedure TMovimientos.RegistrarMovimiento(const AEmpleado: Integer; const
        ADiaHora: TDateTime; const AIDTipo: Integer; AMustSave:
        Boolean);
var
  FMovimiento: TMovimiento;
begin
  FMovimiento:= TMovimiento.CreateInstance(Nil, Self);
  with FMovimiento do begin
     Empleado:= AEmpleado;
     Hora:= ADiaHora;
     IDTipo:= AIDTipo;
  end;
  //aadimos la nueva referencia a la lista de objetos que guarda TMovimientos
  Add(FMovimiento);
  if AMustSave then SaveToFile();
end;

procedure TMovimientos.ReleaseIterator(var it: TIteradorMovimientos);
begin
  it.Free;
  it := nil;
end;

procedure TMovimientos.SaveToFile;
var
  FFilename: String;
begin
  FFilename:= TPath.Combine(TPath.GetHomePath, NOMFILEMOV);
  SaveToFile(FFilename);
end;

procedure TMovimientos.SaveToFile(const AFilename: TFilename);
begin
  if Assigned(FInternal) then begin
     FInternal.SaveToFile(AFilename);
     FModified:= False;
  end;
end;

procedure TMovimientos.SetCadenas(Index: Integer; const Value: string);
begin
  if (Index < 0) or (Index >= FInternal.Count) then
     Raise Exception.Create('Error indice fuera de rango');
  FInternal[Index]:= Value;
  FModified:= True;
end;

procedure TMovimientos.SetItems(Index: integer; Value: TMovimiento);
begin
  if (Index < 0) or (Index >= FInternal.Count) then
     Raise Exception.Create('Error indice fuera de rango');
  with FInternal do begin
     Objects[Index]:= Value;
     if Value <> nil then
        FInternal[Index]:= Value.GetCadenaRegistro
     else FInternal[Index]:= '';
  end;
  FModified:= True;
end;




initialization

   RegisterClasses([TMovimiento]);
   //inicializamos la lista para generar el registro de movimientos
   Lista:= TMovimientos.Create;

finalization

   UnRegisterClasses([TMovimiento]);
   //liberamos la lista de movimientos
   FreeAndNil(Lista);

end.
